<?php
/**
 * Plugin Name: ByteBrew Technologies
 * Plugin URI: https://bbrewtech.com/
 * Description: A professional WordPress PWA plugin with custom install button, no auto-prompts, and mobile-style loader UI. Supports Android Chrome, Desktop Chrome/Edge, and iOS Safari.
 * Version: 1.0.0
 * Author: Your Name
 * Author URI: https://yoursite.com
 * License: GPL v2 or later
 * Text Domain: custom-pwa-installer
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// CRITICAL: Catch service worker request BEFORE WordPress loads
// This intercepts /sw.js requests at the earliest possible point
if (isset($_SERVER['REQUEST_URI'])) {
    $request_uri = $_SERVER['REQUEST_URI'];
    $request_path = strtok($request_uri, '?');
    
    // Check if this is a service worker request
    if (preg_match('#^/sw\.js(\?.*)?$#', $request_uri) || 
        preg_match('#^/sw\.js$#', $request_path) ||
        (isset($_GET['cpwa_service_worker']) && $_GET['cpwa_service_worker'] == '1')) {
        
        // Serve service worker immediately
        $sw_file = __DIR__ . '/assets/js/service-worker.js';
        
        // Clear output buffers
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Disable compression
        if (function_exists('apache_setenv')) {
            @apache_setenv('no-gzip', 1);
        }
        @ini_set('zlib.output_compression', 0);
        
        // Set headers - MUST be 200 OK
        http_response_code(200);
        header('Content-Type: application/javascript; charset=utf-8', true);
        header('Service-Worker-Allowed: /', true);
        header('Cache-Control: no-cache, no-store, must-revalidate, max-age=0', true);
        header('Pragma: no-cache', true);
        header('Expires: 0', true);
        
        // Serve the file
        if (file_exists($sw_file)) {
            readfile($sw_file);
        } else {
            echo "// Service Worker\n";
            echo "self.addEventListener('install', () => self.skipWaiting());\n";
            echo "self.addEventListener('activate', () => self.clients.claim());\n";
        }
        
        exit(0);
    }
}

// Define plugin constants
define('CPWA_VERSION', '1.0.0');
define('CPWA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CPWA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CPWA_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main Plugin Class
 */
class Custom_PWA_Installer {
    
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // Add manifest link to head
        add_action('wp_head', array($this, 'add_manifest_link'), 1);
        
        // Add theme color meta tag
        add_action('wp_head', array($this, 'add_theme_color'), 2);
        
        // Add Apple touch icon
        add_action('wp_head', array($this, 'add_apple_touch_icon'), 3);
        
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        
        // Add install button to footer
        add_action('wp_footer', array($this, 'add_install_button'));
        
        // Register endpoints
        add_action('init', array($this, 'register_manifest_endpoint'));
        add_action('init', array($this, 'register_service_worker_endpoint'));
        add_action('init', array($this, 'register_offline_endpoint'));
        
        // Handle endpoint requests
        add_action('template_redirect', array($this, 'handle_manifest_request'));
        add_action('template_redirect', array($this, 'handle_service_worker_request'));
        add_action('template_redirect', array($this, 'handle_offline_request'));
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Check for HTTPS
        add_action('admin_notices', array($this, 'check_https_notice'));
        
        // Register AJAX handlers
        add_action('wp_ajax_cpwa_get_popup_preview', array($this, 'ajax_get_popup_preview'));
    }
    
    /**
     * Add manifest link to head
     */
    public function add_manifest_link() {
        $manifest_url = home_url('/pwa-manifest.json');
        echo '<link rel="manifest" href="' . esc_url($manifest_url) . '">' . "\n";
    }
    
    /**
     * Add theme color meta tag
     */
    public function add_theme_color() {
        $theme_color = get_option('cpwa_theme_color', '#4285f4');
        echo '<meta name="theme-color" content="' . esc_attr($theme_color) . '">' . "\n";
    }
    
    /**
     * Add Apple touch icon
     */
    public function add_apple_touch_icon() {
        $icon_url = get_option('cpwa_icon_512', CPWA_PLUGIN_URL . 'assets/icons/icon-512x512.png');
        echo '<link rel="apple-touch-icon" href="' . esc_url($icon_url) . '">' . "\n";
        echo '<meta name="mobile-web-app-capable" content="yes">' . "\n";
        echo '<meta name="apple-mobile-web-app-capable" content="yes">' . "\n";
        echo '<meta name="apple-mobile-web-app-status-bar-style" content="default">' . "\n";
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our plugin pages
        $plugin_pages = array(
            'toplevel_page_custom-pwa-installer',
            'pwa-installer_page_cpwa-tools',
            'pwa-installer_page_cpwa-help'
        );
        
        if (!in_array($hook, $plugin_pages)) {
            return;
        }
        
        // Enqueue WordPress media uploader
        wp_enqueue_media();
        
        // Enqueue color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Enqueue PWA install styles for preview
        wp_enqueue_style(
            'cpwa-admin-styles',
            CPWA_PLUGIN_URL . 'assets/css/pwa-install.css',
            array(),
            CPWA_VERSION
        );
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_assets() {
        // Enqueue install button styles
        wp_enqueue_style(
            'cpwa-styles',
            CPWA_PLUGIN_URL . 'assets/css/pwa-install.css',
            array(),
            CPWA_VERSION
        );
        
        // Enqueue install button script
        wp_enqueue_script(
            'cpwa-install',
            CPWA_PLUGIN_URL . 'assets/js/pwa-install.js',
            array(),
            CPWA_VERSION,
            true
        );
        
        // Pass settings to JavaScript
        $is_admin = current_user_can('manage_options');
        
        wp_localize_script('cpwa-install', 'cpwaSettings', array(
            'serviceWorkerUrl' => home_url('/sw.js'),
            'installButtonText' => get_option('cpwa_install_button_text', 'Install App'),
            'installingText' => get_option('cpwa_installing_text', 'Installing...'),
            'installedText' => get_option('cpwa_installed_text', 'App Installed!'),
            'iosMessage' => get_option('cpwa_ios_message', 'To install this app on your iPhone/iPad, tap the Share button and then "Add to Home Screen".'),
            'popupDescription' => get_option('cpwa_popup_description', 'Install our app for a better experience. Get quick access and enjoy offline features.'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cpwa_nonce'),
            'debugMode' => get_option('cpwa_debug_mode', false),
            'popupDelayDays' => absint(get_option('cpwa_popup_delay_days', 0)),
            'popupDelayHours' => absint(get_option('cpwa_popup_delay_hours', 0)),
            'popupDelayMinutes' => absint(get_option('cpwa_popup_delay_minutes', 0)),
            'isAdmin' => $is_admin,
            // Popup customization settings
            'popupBackgroundColor' => get_option('cpwa_popup_background_color', '#ffffff'),
            'popupTextColor' => get_option('cpwa_popup_text_color', '#333333'),
            'popupButtonColor' => get_option('cpwa_popup_button_color', '#667eea'),
            'popupButtonTextColor' => get_option('cpwa_popup_button_text_color', '#ffffff'),
            'popupOverlayColor' => get_option('cpwa_popup_overlay_color', 'rgba(0,0,0,0.6)')
        ));
    }
    
    /**
     * Add install button to footer
     */
    public function add_install_button() {
        if (!get_option('cpwa_show_install_button', true)) {
            return;
        }
        
        ?>
        <!-- PWA Install Popup Modal -->
        <div id="cpwa-install-popup" class="cpwa-install-popup cpwa-hidden">
            <div class="cpwa-popup-overlay"></div>
            <div class="cpwa-popup-content">
                <button id="cpwa-popup-close" class="cpwa-popup-close" aria-label="Close install prompt">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"></line>
                        <line x1="6" y1="6" x2="18" y2="18"></line>
                    </svg>
                </button>
                <div class="cpwa-popup-body">
                    <div class="cpwa-popup-icon">
                        <?php 
                        $popup_icon = get_option('cpwa_icon_192') ?: get_option('cpwa_icon_512', '');
                        if ($popup_icon) : ?>
                            <img src="<?php echo esc_url($popup_icon); ?>" alt="<?php echo esc_attr(get_option('cpwa_app_name', get_bloginfo('name'))); ?>" style="width: 100%; height: 100%; object-fit: contain; border-radius: 20px;">
                        <?php else : ?>
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                <polyline points="7 10 12 15 17 10"></polyline>
                                <line x1="12" y1="15" x2="12" y2="3"></line>
                            </svg>
                        <?php endif; ?>
                    </div>
                    <h3 class="cpwa-popup-title"><?php echo esc_html(get_option('cpwa_install_button_text', 'Install App')); ?></h3>
                    <p class="cpwa-popup-description"><?php echo esc_html(get_option('cpwa_popup_description', 'Install our app for a better experience. Get quick access and enjoy offline features.')); ?></p>
                    <button id="cpwa-install-button" class="cpwa-install-btn">
                        <svg class="cpwa-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="7 10 12 15 17 10"></polyline>
                            <line x1="12" y1="15" x2="12" y2="3"></line>
                        </svg>
                        <span class="cpwa-button-text"><?php echo esc_html(get_option('cpwa_install_button_text', 'Install App')); ?></span>
                    </button>
                </div>
            </div>
        </div>
        
        <!-- PWA Loader Overlay -->
        <div id="cpwa-loader-overlay" class="cpwa-loader-overlay cpwa-hidden">
            <div class="cpwa-loader-content">
                <div class="cpwa-loader-spinner">
                    <div class="cpwa-spinner-ring"></div>
                    <div class="cpwa-spinner-ring"></div>
                    <div class="cpwa-spinner-ring"></div>
                </div>
                <p class="cpwa-loader-text"><?php echo esc_html(get_option('cpwa_installing_text', 'Installing...')); ?></p>
                <p class="cpwa-loader-subtext">Please wait while we set up the app</p>
            </div>
        </div>
        
        <!-- iOS Install Instructions -->
        <div id="cpwa-ios-modal" class="cpwa-modal cpwa-hidden">
            <div class="cpwa-modal-overlay"></div>
            <div class="cpwa-modal-content">
                <button class="cpwa-modal-close">&times;</button>
                <div class="cpwa-modal-body">
                    <svg class="cpwa-ios-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <rect x="5" y="2" width="14" height="20" rx="2" ry="2"></rect>
                        <line x1="12" y1="18" x2="12.01" y2="18"></line>
                    </svg>
                    <h3>Install App</h3>
                    <p><?php echo esc_html(get_option('cpwa_ios_message', 'To install this app on your iPhone/iPad, tap the Share button and then "Add to Home Screen".')); ?></p>
                    <div class="cpwa-ios-steps">
                        <div class="cpwa-step">
                            <span class="cpwa-step-number">1</span>
                            <span>Tap the <strong>Share</strong> button (
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M16 5l-1.42 1.42-1.59-1.59V16h-1.98V4.83L9.42 6.42 8 5l4-4 4 4zm4 5v11c0 1.1-.9 2-2 2H6c-1.11 0-2-.9-2-2V10c0-1.11.89-2 2-2h3v2H6v11h12V10h-3V8h3c1.1 0 2 .89 2 2z"/>
                                </svg>
                            )</span>
                        </div>
                        <div class="cpwa-step">
                            <span class="cpwa-step-number">2</span>
                            <span>Select <strong>"Add to Home Screen"</strong></span>
                        </div>
                        <div class="cpwa-step">
                            <span class="cpwa-step-number">3</span>
                            <span>Tap <strong>"Add"</strong></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Register manifest endpoint
     */
    public function register_manifest_endpoint() {
        add_rewrite_rule('^pwa-manifest\.json$', 'index.php?cpwa_manifest=1', 'top');
        add_rewrite_tag('%cpwa_manifest%', '([^&]+)');
    }
    
    /**
     * Register service worker endpoint
     */
    public function register_service_worker_endpoint() {
        add_rewrite_rule('^sw\.js$', 'index.php?cpwa_service_worker=1', 'top');
        add_rewrite_tag('%cpwa_service_worker%', '([^&]+)');
        
        // Add query var to ensure it's recognized
        add_filter('query_vars', function($vars) {
            if (!in_array('cpwa_service_worker', $vars)) {
                $vars[] = 'cpwa_service_worker';
            }
            return $vars;
        });
    }
    
    /**
     * Register offline page endpoint
     */
    public function register_offline_endpoint() {
        add_rewrite_rule('^offline\.html$', 'index.php?cpwa_offline=1', 'top');
        add_rewrite_tag('%cpwa_offline%', '([^&]+)');
    }
    
    /**
     * Handle manifest request
     */
    public function handle_manifest_request() {
        if (get_query_var('cpwa_manifest')) {
            header('Content-Type: application/json');
            header('Service-Worker-Allowed: /');
            
            $manifest = $this->generate_manifest();
            echo wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            exit;
        }
    }
    
    /**
     * Generate manifest data with enhanced PWA compliance
     */
    private function generate_manifest() {
        $site_name = get_option('cpwa_app_name', get_bloginfo('name'));
        $short_name = get_option('cpwa_short_name', get_bloginfo('name'));
        $description = get_option('cpwa_description', get_bloginfo('description'));
        $theme_color = get_option('cpwa_theme_color', '#4285f4');
        $background_color = get_option('cpwa_background_color', '#ffffff');
        $start_url = get_option('cpwa_start_url', home_url('/'));
        
        // Ensure short name is max 12 characters for better PWA compliance
        if (strlen($short_name) > 12) {
            $short_name = substr($short_name, 0, 12);
        }
        
        // Generate manifest with all required PWA fields
        $manifest = array(
            'name' => $site_name,
            'short_name' => $short_name,
            'description' => $description,
            'start_url' => $start_url,
            'display' => 'standalone',
            'orientation' => 'any',
            'theme_color' => $theme_color,
            'background_color' => $background_color,
            'scope' => '/',
            'lang' => get_locale(),
            'dir' => is_rtl() ? 'rtl' : 'ltr',
            'prefer_related_applications' => false,
            'icons' => $this->get_manifest_icons()
        );
        
        // Add categories if available
        $categories = get_option('cpwa_app_categories', array());
        if (!empty($categories)) {
            $manifest['categories'] = $categories;
        }
        
        return $manifest;
    }
    
    /**
     * Get manifest icons with comprehensive size support
     */
    private function get_manifest_icons() {
        $icons = array();
        
        // Get custom icons or use defaults
        $icon_192 = get_option('cpwa_icon_192') ?: CPWA_PLUGIN_URL . 'assets/icons/icon-192x192.png';
        $icon_512 = get_option('cpwa_icon_512') ?: CPWA_PLUGIN_URL . 'assets/icons/icon-512x512.png';
        
        // Define required icon sizes for better PWA support
        $icon_sizes = array(
            '72' => $icon_192,
            '96' => $icon_192,
            '128' => $icon_192,
            '144' => $icon_192,
            '152' => $icon_192,
            '192' => $icon_192,
            '384' => $icon_512,
            '512' => $icon_512
        );
        
        // Generate icon array with proper purposes
        foreach ($icon_sizes as $size => $src) {
            $icons[] = array(
                'src' => $src,
                'sizes' => $size . 'x' . $size,
                'type' => 'image/png',
                'purpose' => in_array($size, array('192', '512')) ? 'any maskable' : 'any'
            );
        }
        
        return $icons;
    }
    
    /**
     * Handle service worker request
     */
    public function handle_service_worker_request() {
        // Check if query var is set (from rewrite rule)
        if (get_query_var('cpwa_service_worker')) {
            $this->serve_service_worker();
            return;
        }
        
        // Also check REQUEST_URI directly as fallback
        if (isset($_SERVER['REQUEST_URI'])) {
            $request_uri = $_SERVER['REQUEST_URI'];
            $request_path = strtok($request_uri, '?');
            
            if (preg_match('#^/sw\.js(\?.*)?$#', $request_uri) || preg_match('#^/sw\.js$#', $request_path)) {
                $this->serve_service_worker();
            }
        }
    }
    
    /**
     * Serve service worker file directly
     */
    private function serve_service_worker() {
        // Clear output buffers
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Disable compression
        if (function_exists('apache_setenv')) {
            @apache_setenv('no-gzip', 1);
        }
        @ini_set('zlib.output_compression', 0);
        
        // Set proper headers - MUST be 200 OK
        http_response_code(200);
        status_header(200);
        header('Content-Type: application/javascript; charset=utf-8', true);
        header('Service-Worker-Allowed: /', true);
        header('Cache-Control: no-cache, no-store, must-revalidate, max-age=0', true);
        header('Pragma: no-cache', true);
        header('Expires: 0', true);
        
        // Prevent WordPress from adding any output
        remove_all_actions('wp_head');
        remove_all_actions('wp_footer');
        remove_all_actions('shutdown');
        
        // Serve the service worker file
        $sw_file = CPWA_PLUGIN_DIR . 'assets/js/service-worker.js';
        if (file_exists($sw_file)) {
            readfile($sw_file);
        } else {
            // Fallback minimal service worker
            echo "// Service Worker\n";
            echo "self.addEventListener('install', () => self.skipWaiting());\n";
            echo "self.addEventListener('activate', () => self.clients.claim());\n";
        }
        
        exit(0);
    }
    
    /**
     * Handle offline page request
     */
    public function handle_offline_request() {
        if (get_query_var('cpwa_offline')) {
            header('Content-Type: text/html; charset=utf-8');
            
            if (file_exists(CPWA_PLUGIN_DIR . 'offline.html')) {
                include CPWA_PLUGIN_DIR . 'offline.html';
            } else {
                echo '<!DOCTYPE html><html><head><title>Offline</title></head><body><h1>You are offline</h1><p>Please check your internet connection.</p></body></html>';
            }
            exit;
        }
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Add top-level menu
        add_menu_page(
            __('PWA Installer', 'custom-pwa-installer'),
            __('PWA Installer', 'custom-pwa-installer'),
            'manage_options',
            'custom-pwa-installer',
            array($this, 'render_admin_page'),
            'dashicons-smartphone',
            80
        );
        
        // Add Settings submenu
        add_submenu_page(
            'custom-pwa-installer',
            __('PWA Settings', 'custom-pwa-installer'),
            __('Settings', 'custom-pwa-installer'),
            'manage_options',
            'custom-pwa-installer',
            array($this, 'render_admin_page')
        );
        
        // Add Tools submenu
        add_submenu_page(
            'custom-pwa-installer',
            __('PWA Tools', 'custom-pwa-installer'),
            __('Tools', 'custom-pwa-installer'),
            'manage_options',
            'cpwa-tools',
            array($this, 'render_tools_page')
        );
        
        // Add Help submenu
        add_submenu_page(
            'custom-pwa-installer',
            __('PWA Help', 'custom-pwa-installer'),
            __('Help', 'custom-pwa-installer'),
            'manage_options',
            'cpwa-help',
            array($this, 'render_help_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // General settings
        register_setting('cpwa_settings', 'cpwa_app_name', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('cpwa_settings', 'cpwa_short_name', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('cpwa_settings', 'cpwa_description', array('sanitize_callback' => 'sanitize_textarea_field'));
        register_setting('cpwa_settings', 'cpwa_theme_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_background_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_start_url', array('sanitize_callback' => 'esc_url_raw'));
        register_setting('cpwa_settings', 'cpwa_show_install_button', array('sanitize_callback' => 'absint'));
        
        // Button text settings
        register_setting('cpwa_settings', 'cpwa_install_button_text', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('cpwa_settings', 'cpwa_installing_text', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('cpwa_settings', 'cpwa_installed_text', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('cpwa_settings', 'cpwa_ios_message', array('sanitize_callback' => 'sanitize_textarea_field'));
        register_setting('cpwa_settings', 'cpwa_popup_description', array('sanitize_callback' => 'sanitize_textarea_field'));
        register_setting('cpwa_settings', 'cpwa_popup_delay_days', array('sanitize_callback' => 'absint'));
        register_setting('cpwa_settings', 'cpwa_popup_delay_hours', array('sanitize_callback' => 'absint'));
        register_setting('cpwa_settings', 'cpwa_popup_delay_minutes', array('sanitize_callback' => 'absint'));
        
        // Popup appearance settings
        register_setting('cpwa_settings', 'cpwa_popup_background_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_popup_text_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_popup_button_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_popup_button_text_color', array('sanitize_callback' => 'sanitize_hex_color'));
        register_setting('cpwa_settings', 'cpwa_popup_overlay_color', array('sanitize_callback' => 'sanitize_hex_color'));
        
        // Icon settings
        register_setting('cpwa_settings', 'cpwa_icon_192', array('sanitize_callback' => 'esc_url_raw'));
        register_setting('cpwa_settings', 'cpwa_icon_512', array('sanitize_callback' => 'esc_url_raw'));
        
        // Debug settings
        register_setting('cpwa_settings', 'cpwa_debug_mode', array('sanitize_callback' => 'absint'));
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Save settings
        if (isset($_POST['cpwa_save_settings']) && check_admin_referer('cpwa_settings_nonce', 'cpwa_settings_nonce_field')) {
            echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved successfully!', 'custom-pwa-installer') . '</p></div>';
        }
        
        include CPWA_PLUGIN_DIR . 'templates/admin-page.php';
    }
    
    /**
     * Render tools page
     */
    public function render_tools_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Handle cache clearing
        if (isset($_POST['cpwa_clear_cache']) && check_admin_referer('cpwa_tools_nonce', 'cpwa_tools_nonce_field')) {
            echo '<div class="notice notice-success"><p>' . esc_html__('Cache clearing instruction sent to service worker!', 'custom-pwa-installer') . '</p></div>';
        }
        
        include CPWA_PLUGIN_DIR . 'templates/tools-page.php';
    }
    
    /**
     * Render help page
     */
    public function render_help_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        include CPWA_PLUGIN_DIR . 'templates/help-page.php';
    }
    
    /**
     * Check HTTPS notice
     */
    public function check_https_notice() {
        if (!is_ssl() && current_user_can('manage_options')) {
            ?>
            <div class="notice notice-error">
                <p><strong><?php esc_html_e('PWA Warning:', 'custom-pwa-installer'); ?></strong> 
                <?php esc_html_e('Your site must use HTTPS for PWA features to work. Service workers require a secure connection.', 'custom-pwa-installer'); ?>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * AJAX handler to get popup preview HTML
     */
    public function ajax_get_popup_preview() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'custom-pwa-installer')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'cpwa_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'custom-pwa-installer')));
            return;
        }
        
        // Get current settings (or use provided values for preview)
        $popup_bg = isset($_POST['popup_bg']) ? sanitize_hex_color($_POST['popup_bg']) : get_option('cpwa_popup_background_color', '#ffffff');
        $popup_text = isset($_POST['popup_text']) ? sanitize_hex_color($_POST['popup_text']) : get_option('cpwa_popup_text_color', '#333333');
        $popup_button = isset($_POST['popup_button']) ? sanitize_hex_color($_POST['popup_button']) : get_option('cpwa_popup_button_color', '#667eea');
        $popup_button_text = isset($_POST['popup_button_text']) ? sanitize_hex_color($_POST['popup_button_text']) : get_option('cpwa_popup_button_text_color', '#ffffff');
        $popup_overlay = isset($_POST['popup_overlay']) ? sanitize_text_field($_POST['popup_overlay']) : get_option('cpwa_popup_overlay_color', 'rgba(0,0,0,0.6)');
        $button_text = isset($_POST['button_text']) ? sanitize_text_field($_POST['button_text']) : get_option('cpwa_install_button_text', 'Install App');
        $description = isset($_POST['description']) ? sanitize_textarea_field($_POST['description']) : get_option('cpwa_popup_description', 'Install our app for a better experience. Get quick access and enjoy offline features.');
        $icon_192 = isset($_POST['icon_192']) ? esc_url_raw($_POST['icon_192']) : get_option('cpwa_icon_192', '');
        $icon_512 = isset($_POST['icon_512']) ? esc_url_raw($_POST['icon_512']) : get_option('cpwa_icon_512', '');
        $popup_icon = $icon_192 ?: $icon_512;
        
        // Generate popup HTML
        ob_start();
        ?>
        <div class="cpwa-install-popup cpwa-visible" style="position: relative; z-index: 1;">
            <div class="cpwa-popup-overlay" style="background: <?php echo esc_attr($popup_overlay); ?>;"></div>
            <div class="cpwa-popup-content" style="background: <?php echo esc_attr($popup_bg); ?>;">
                <button class="cpwa-popup-close cpwa-preview-close" aria-label="Close preview" style="position: absolute; top: 15px; right: 15px; width: 36px; height: 36px; border: none; background: #f5f5f5; border-radius: 50%; font-size: 24px; line-height: 36px; cursor: pointer; color: #666666; transition: all 0.2s ease; padding: 0; display: flex; align-items: center; justify-content: center; z-index: 10;">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 18px; height: 18px;">
                        <line x1="18" y1="6" x2="6" y2="18"></line>
                        <line x1="6" y1="6" x2="18" y2="18"></line>
                    </svg>
                </button>
                <div class="cpwa-popup-body" style="padding: 40px 30px; text-align: center;">
                    <div class="cpwa-popup-icon" style="width: 80px; height: 80px; margin: 0 auto 20px; background: <?php echo $popup_icon ? 'transparent' : 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)'; ?>; border-radius: 20px; display: flex; align-items: center; justify-content: center; box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3); overflow: hidden;">
                        <?php if ($popup_icon) : ?>
                            <img src="<?php echo esc_url($popup_icon); ?>" alt="App Icon" style="width: 100%; height: 100%; object-fit: contain; border-radius: 20px; display: block;">
                        <?php else : ?>
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 40px; height: 40px; color: #ffffff;">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                <polyline points="7 10 12 15 17 10"></polyline>
                                <line x1="12" y1="15" x2="12" y2="3"></line>
                            </svg>
                        <?php endif; ?>
                    </div>
                    <h3 class="cpwa-popup-title" style="font-size: 24px; font-weight: 700; color: <?php echo esc_attr($popup_text); ?>; margin: 0 0 12px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;">
                        <?php echo esc_html($button_text); ?>
                    </h3>
                    <p class="cpwa-popup-description" style="font-size: 16px; color: <?php echo esc_attr($popup_text); ?>; opacity: 0.8; line-height: 1.6; margin: 0 0 30px 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;">
                        <?php echo esc_html($description); ?>
                    </p>
                    <button class="cpwa-install-btn" style="display: flex; align-items: center; justify-content: center; gap: 10px; padding: 16px 32px; background: <?php echo esc_attr($popup_button); ?>; color: <?php echo esc_attr($popup_button_text); ?>; border: none; border-radius: 50px; font-size: 16px; font-weight: 600; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif; cursor: pointer; box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4); transition: all 0.3s ease; text-decoration: none; outline: none; width: 100%;">
                        <svg class="cpwa-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="width: 20px; height: 20px;">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="7 10 12 15 17 10"></polyline>
                            <line x1="12" y1="15" x2="12" y2="3"></line>
                        </svg>
                        <span class="cpwa-button-text"><?php echo esc_html($button_text); ?></span>
                    </button>
                </div>
            </div>
        </div>
        <?php
        $html = ob_get_clean();
        
        wp_send_json_success(array(
            'html' => $html,
            'settings' => array(
                'popup_bg' => $popup_bg,
                'popup_text' => $popup_text,
                'popup_button' => $popup_button,
                'popup_button_text' => $popup_button_text,
                'popup_overlay' => $popup_overlay
            )
        ));
    }
}

// Initialize plugin
function cpwa_init() {
    return Custom_PWA_Installer::get_instance();
}
add_action('plugins_loaded', 'cpwa_init');

// Activation hook - flush rewrite rules
register_activation_hook(__FILE__, 'cpwa_activate');
function cpwa_activate() {
    // Register endpoints
    $plugin = Custom_PWA_Installer::get_instance();
    $plugin->register_manifest_endpoint();
    $plugin->register_service_worker_endpoint();
    $plugin->register_offline_endpoint();
    
    // Flush rewrite rules
    flush_rewrite_rules();
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'cpwa_deactivate');
function cpwa_deactivate() {
    // Flush rewrite rules
    flush_rewrite_rules();
}
