/**
 * Service Worker for Custom PWA Installer
 * Version: 1.0.0
 * 
 * This service worker provides offline functionality and caching
 * for the WordPress PWA application.
 */

const CACHE_VERSION = 'cpwa-v1.0.0';
const CACHE_NAME = `cpwa-cache-${CACHE_VERSION}`;

// Assets to cache immediately on install
const PRECACHE_URLS = [
    '/',
    '/offline.html'
];

// Cache strategies
const CACHE_STRATEGIES = {
    images: 'cache-first',
    styles: 'cache-first',
    scripts: 'cache-first',
    pages: 'network-first',
    api: 'network-only'
};

/**
 * Install Event
 * Fired when service worker is first installed
 */
self.addEventListener('install', (event) => {
    console.log('[Service Worker] Installing service worker...', event);
    
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                console.log('[Service Worker] Precaching app shell');
                // Try to cache URLs, but don't fail if some fail
                return Promise.allSettled(
                    PRECACHE_URLS.map(url => {
                        return cache.add(new Request(url, {cache: 'reload'})).catch((err) => {
                            console.warn('[Service Worker] Failed to cache:', url, err);
                            return null; // Don't fail the whole precache
                        });
                    })
                );
            })
            .then(() => {
                // Force the waiting service worker to become the active service worker
                return self.skipWaiting();
            })
            .catch((error) => {
                console.error('[Service Worker] Precaching failed:', error);
                // Still skip waiting even if precaching fails
                return self.skipWaiting();
            })
    );
});

/**
 * Activate Event
 * Fired when service worker is activated
 */
self.addEventListener('activate', (event) => {
    console.log('[Service Worker] Activating service worker...', event);
    
    event.waitUntil(
        caches.keys()
            .then((cacheNames) => {
                // Delete old caches
                return Promise.all(
                    cacheNames.map((cacheName) => {
                        if (cacheName !== CACHE_NAME && cacheName.startsWith('cpwa-cache-')) {
                            console.log('[Service Worker] Deleting old cache:', cacheName);
                            return caches.delete(cacheName);
                        }
                    })
                );
            })
            .then(() => {
                // Take control of all clients immediately
                return self.clients.claim();
            })
    );
});

/**
 * Enhanced Fetch Event
 * Intercept network requests and serve from cache with intelligent fallbacks
 */
self.addEventListener('fetch', (event) => {
    const { request } = event;
    const url = new URL(request.url);
    
    // Skip non-GET requests
    if (request.method !== 'GET') {
        return;
    }
    
    // Skip chrome-extension and other non-http(s) requests
    if (!url.protocol.startsWith('http')) {
        return;
    }
    
    // Skip admin and wp-login pages (don't cache sensitive content)
    if (url.pathname.includes('/wp-admin/') || url.pathname.includes('/wp-login.php')) {
        return;
    }
    
    // Skip external CDN requests that might have CORS issues
    if (url.origin !== self.location.origin && !url.hostname.includes('wordpress.org')) {
        return;
    }
    
    // Determine caching strategy based on request type
    const strategy = getCachingStrategy(request);
    
    event.respondWith(
        handleFetchWithStrategy(request, strategy)
            .catch((error) => {
                console.error('[Service Worker] Fetch failed, trying fallback:', error);
                return handleFetchError(request, error);
            })
    );
});

/**
 * Handle fetch errors with appropriate fallbacks
 */
function handleFetchError(request, error) {
    const url = new URL(request.url);
    
    // For navigation requests, serve offline page
    if (request.destination === 'document') {
        return caches.match('/offline.html').then((response) => {
            if (response) {
                console.log('[Service Worker] Serving offline page for:', request.url);
                return response;
            }
            // Fallback to basic offline response
            return new Response('Offline - Please check your connection', {
                status: 503,
                statusText: 'Service Unavailable',
                headers: { 'Content-Type': 'text/html' }
            });
        });
    }
    
    // For other requests, try to serve from cache
    return caches.match(request).then((response) => {
        if (response) {
            console.log('[Service Worker] Serving from cache fallback:', request.url);
            return response;
        }
        
        // Final fallback - return error response
        return new Response('Network error', {
            status: 408,
            statusText: 'Request Timeout',
            headers: { 'Content-Type': 'text/plain' }
        });
    });
}

/**
 * Determine caching strategy based on request
 */
function getCachingStrategy(request) {
    const url = new URL(request.url);
    const pathname = url.pathname;
    const destination = request.destination;
    
    // Images
    if (destination === 'image' || /\.(jpg|jpeg|png|gif|svg|webp|ico)$/i.test(pathname)) {
        return CACHE_STRATEGIES.images;
    }
    
    // Styles
    if (destination === 'style' || /\.css$/i.test(pathname)) {
        return CACHE_STRATEGIES.styles;
    }
    
    // Scripts
    if (destination === 'script' || /\.js$/i.test(pathname)) {
        return CACHE_STRATEGIES.scripts;
    }
    
    // API requests
    if (pathname.includes('/wp-json/') || pathname.includes('admin-ajax.php')) {
        return CACHE_STRATEGIES.api;
    }
    
    // Default to network-first for pages
    return CACHE_STRATEGIES.pages;
}

/**
 * Handle fetch with specific caching strategy
 */
async function handleFetchWithStrategy(request, strategy) {
    switch (strategy) {
        case 'cache-first':
            return cacheFirst(request);
        case 'network-first':
            return networkFirst(request);
        case 'network-only':
            return networkOnly(request);
        default:
            return fetch(request);
    }
}

/**
 * Cache First Strategy
 * Try cache first, fall back to network
 */
async function cacheFirst(request) {
    const cache = await caches.open(CACHE_NAME);
    const cachedResponse = await cache.match(request);
    
    if (cachedResponse) {
        console.log('[Service Worker] Cache hit:', request.url);
        return cachedResponse;
    }
    
    try {
        const networkResponse = await fetch(request);
        
        // Cache successful responses
        if (networkResponse && networkResponse.status === 200) {
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.error('[Service Worker] Fetch failed:', error);
        
        // Return offline page for navigation requests
        if (request.destination === 'document') {
            const offlineResponse = await cache.match('/offline.html');
            if (offlineResponse) {
                return offlineResponse;
            }
        }
        
        throw error;
    }
}

/**
 * Network First Strategy
 * Try network first, fall back to cache
 */
async function networkFirst(request) {
    const cache = await caches.open(CACHE_NAME);
    
    try {
        const networkResponse = await fetch(request);
        
        // Cache successful responses
        if (networkResponse && networkResponse.status === 200) {
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.log('[Service Worker] Network failed, trying cache:', request.url);
        
        const cachedResponse = await cache.match(request);
        if (cachedResponse) {
            return cachedResponse;
        }
        
        // Return offline page for navigation requests
        if (request.destination === 'document') {
            const offlineResponse = await cache.match('/offline.html');
            if (offlineResponse) {
                return offlineResponse;
            }
        }
        
        throw error;
    }
}

/**
 * Network Only Strategy
 * Always fetch from network
 */
async function networkOnly(request) {
    return fetch(request);
}

/**
 * Message Event
 * Handle messages from clients
 */
self.addEventListener('message', (event) => {
    console.log('[Service Worker] Message received:', event.data);
    
    if (event.data && event.data.type === 'SKIP_WAITING') {
        self.skipWaiting();
    }
    
    if (event.data && event.data.type === 'CACHE_URLS') {
        const urls = event.data.urls || [];
        caches.open(CACHE_NAME).then((cache) => {
            cache.addAll(urls);
        });
    }
    
    if (event.data && event.data.type === 'CLEAR_CACHE') {
        caches.keys().then((cacheNames) => {
            return Promise.all(
                cacheNames.map((cacheName) => {
                    if (cacheName.startsWith('cpwa-cache-')) {
                        return caches.delete(cacheName);
                    }
                })
            );
        });
    }
});

/**
 * Push Event (for future push notifications support)
 */
self.addEventListener('push', (event) => {
    console.log('[Service Worker] Push notification received:', event);
    
    const options = {
        body: event.data ? event.data.text() : 'New notification',
        icon: '/assets/icons/icon-192x192.png',
        badge: '/assets/icons/icon-192x192.png',
        vibrate: [100, 50, 100],
        data: {
            dateOfArrival: Date.now(),
            primaryKey: 1
        }
    };
    
    event.waitUntil(
        self.registration.showNotification('PWA Notification', options)
    );
});

/**
 * Notification Click Event
 */
self.addEventListener('notificationclick', (event) => {
    console.log('[Service Worker] Notification clicked:', event);
    
    event.notification.close();
    
    event.waitUntil(
        clients.openWindow('/')
    );
});

console.log('[Service Worker] Service worker script loaded');

