/**
 * PWA Install Button Handler
 * Handles the beforeinstallprompt event and custom install button
 * 
 * Features:
 * - Prevents auto-prompt
 * - Shows custom install button only when installable
 * - Mobile-style loader UI on install
 * - iOS Safari detection with manual instructions
 * - Cross-browser support (Chrome, Edge, Safari)
 * - Re-show logic with configurable delay after dismissal (days, hours, minutes)
 */

(function() {
    'use strict';
    
    const STORAGE_KEY_DISMISSED = 'cpwa_dismissed_timestamp';
    const MILLISECONDS_PER_MINUTE = 60 * 1000;
    const MILLISECONDS_PER_HOUR = 60 * MILLISECONDS_PER_MINUTE;
    const MILLISECONDS_PER_DAY = 24 * MILLISECONDS_PER_HOUR;
    
    let deferredPrompt = null;
    let isInstalled = false;
    let isInstalling = false;
    
    let installPopup = null;
    let installButton = null;
    let popupCloseButton = null;
    let loaderOverlay = null;
    let iosModal = null;
    let iosModalClose = null;
    
    /**
     * Initialize the PWA install functionality
     */
    function init() {
        if (typeof cpwaSettings === 'undefined') {
            return;
        }
        
        installPopup = document.getElementById('cpwa-install-popup');
        installButton = document.getElementById('cpwa-install-button');
        popupCloseButton = document.getElementById('cpwa-popup-close');
        loaderOverlay = document.getElementById('cpwa-loader-overlay');
        iosModal = document.getElementById('cpwa-ios-modal');
        iosModalClose = iosModal ? iosModal.querySelector('.cpwa-modal-close') : null;
        
        if (!installPopup) {
            return;
        }
        
        if (isStandalone()) {
            isInstalled = true;
            hideInstallButton();
            return;
        }
        
        registerServiceWorker();
        setupEventListeners();
        
        if (!shouldShowInstallPopup()) {
            hideInstallButton();
            return;
        }
        
        if (cpwaSettings.debugMode) {
            showInstallButton();
            return;
        }
        
        if (isIOS() && !isStandalone()) {
            showInstallButton();
            return;
        }
        
        if (deferredPrompt) {
            showInstallButton();
        }
    }
    
    /**
     * Register service worker
     */
    function registerServiceWorker() {
        if (!('serviceWorker' in navigator)) {
            return;
        }
        
        const serviceWorkerUrl = cpwaSettings.serviceWorkerUrl || '/sw.js';
        
        let swUrl = serviceWorkerUrl;
        if (!swUrl.startsWith('http')) {
            swUrl = window.location.origin + (swUrl.startsWith('/') ? swUrl : '/' + swUrl);
        }
        
        navigator.serviceWorker.register(swUrl, { scope: '/' })
            .then((registration) => {
                registration.addEventListener('updatefound', () => {
                    const newWorker = registration.installing;
                    if (newWorker) {
                        newWorker.addEventListener('statechange', () => {
                            if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                            }
                        });
                    }
                });
            })
            .catch((error) => {
            });
    }
    
    /**
     * Setup event listeners
     */
    function setupEventListeners() {
        window.addEventListener('beforeinstallprompt', (e) => {
            if (isStandalone()) {
                return;
            }
            
            e.preventDefault();
            deferredPrompt = e;
            showInstallButton();
        });
        
        window.addEventListener('appinstalled', (e) => {
            isInstalled = true;
            hideLoader();
            hideInstallButton();
            deferredPrompt = null;
            showSuccessMessage();
        });
        
        if (installButton) {
            installButton.addEventListener('click', handleInstallClick);
        }
        
        if (popupCloseButton) {
            popupCloseButton.addEventListener('click', handleCloseClick);
        }
        
        if (installPopup) {
            const overlay = installPopup.querySelector('.cpwa-popup-overlay');
            if (overlay) {
                overlay.addEventListener('click', handleCloseClick);
            }
        }
        
        if (iosModalClose) {
            iosModalClose.addEventListener('click', closeIOSModal);
        }
        
        if (iosModal) {
            const overlay = iosModal.querySelector('.cpwa-modal-overlay');
            if (overlay) {
                overlay.addEventListener('click', closeIOSModal);
            }
        }
    }
    
    /**
     * Handle install button click
     */
    function handleInstallClick() {
        if (isInstalling) {
            return;
        }
        
        if (isIOS()) {
            showIOSModal();
            return;
        }
        
        if (!deferredPrompt) {
            return;
        }
        
        isInstalling = true;
        showLoader();
        hideInstallButton();
        deferredPrompt.prompt();
        
        deferredPrompt.userChoice.then((choiceResult) => {
            if (choiceResult.outcome === 'accepted') {
            } else {
                hideLoader();
                isInstalling = false;
                setPopupDismissalTimestamp();
            }
            
            deferredPrompt = null;
        }).catch((error) => {
            hideLoader();
            isInstalling = false;
            setPopupDismissalTimestamp();
        });
    }
    
    /**
     * Show install popup
     */
    function showInstallButton() {
        if (installPopup && !isStandalone()) {
            installPopup.classList.remove('cpwa-hidden');
            installPopup.classList.add('cpwa-visible');
            document.body.style.overflow = 'hidden';
        }
    }
    
    /**
     * Hide install popup
     */
    function hideInstallButton() {
        if (installPopup) {
            installPopup.classList.remove('cpwa-visible');
            installPopup.classList.add('cpwa-hidden');
            document.body.style.overflow = '';
        }
    }
    
    /**
     * Handle close button click
     * Stores dismissal timestamp and hides the install popup
     */
    function handleCloseClick(e) {
        e.stopPropagation();
        setPopupDismissalTimestamp();
        setSessionDismissed();
        hideInstallButton();
    }
    
    /**
     * Check if popup was dismissed in current session
     * 
     * @returns {boolean} True if dismissed in current session, false otherwise
     */
    function isSessionDismissed() {
        try {
            const sessionDismissed = sessionStorage.getItem('cpwa_dismissed_session');
            return sessionDismissed === 'true';
        } catch (error) {
            return false;
        }
    }
    
    /**
     * Set session dismissal flag
     */
    function setSessionDismissed() {
        try {
            sessionStorage.setItem('cpwa_dismissed_session', 'true');
            sessionStorage.setItem('cpwa_dismissal_time', Date.now().toString());
        } catch (error) {
        }
    }
    
    /**
     * Determines if the install popup should be shown
     * 
     * Returns true if:
     * - No dismissal timestamp exists (user never closed it)
     * - Enough time has passed since dismissal
     * 
     * Returns false if:
     * - App is already installed
     * - User dismissed it in current session
     * - User dismissed it and delay period hasn't expired
     * 
     * @returns {boolean} True if popup should be shown, false otherwise
     */
    function shouldShowInstallPopup() {
        if (isStandalone()) {
            return false;
        }
        
        if (isSessionDismissed()) {
            return false;
        }
        
        const dismissedTimestamp = getPopupDismissalTimestamp();
        
        if (!dismissedTimestamp) {
            return true;
        }
        
        const timeSinceDismissal = Date.now() - dismissedTimestamp;
        const delayMs = getPopupDelayMilliseconds();
        
        if (timeSinceDismissal >= delayMs) {
            try {
                sessionStorage.removeItem('cpwa_dismissed_session');
                sessionStorage.removeItem('cpwa_dismissal_time');
            } catch (error) {
            }
            return true;
        }
        
        return false;
    }
    
    /**
     * Get the popup dismissal timestamp from localStorage
     * 
     * @returns {number|null} Timestamp in milliseconds, or null if not found
     */
    function getPopupDismissalTimestamp() {
        try {
            const timestamp = localStorage.getItem(STORAGE_KEY_DISMISSED);
            
            if (!timestamp) {
                return null;
            }
            
            const parsedTimestamp = parseInt(timestamp, 10);
            
            if (isNaN(parsedTimestamp) || parsedTimestamp <= 0) {
                clearPopupDismissalTimestamp();
                return null;
            }
            
            return parsedTimestamp;
        } catch (error) {
            return null;
        }
    }
    
    /**
     * Store the current timestamp in localStorage
     * Called when user clicks the close button
     */
    function setPopupDismissalTimestamp() {
        try {
            const currentTimestamp = Date.now();
            localStorage.setItem(STORAGE_KEY_DISMISSED, currentTimestamp.toString());
        } catch (error) {
        }
    }
    
    /**
     * Clear the dismissal timestamp from localStorage
     */
    function clearPopupDismissalTimestamp() {
        try {
            localStorage.removeItem(STORAGE_KEY_DISMISSED);
        } catch (error) {
        }
    }
    
    /**
     * Get the configured delay in milliseconds from WordPress settings
     * Supports days, hours, and minutes
     * 
     * @returns {number} Number of milliseconds to wait before re-showing popup
     */
    function getPopupDelayMilliseconds() {
        if (typeof cpwaSettings === 'undefined') {
            return MILLISECONDS_PER_DAY;
        }
        
        const delayDays = parseInt(cpwaSettings.popupDelayDays || 0, 10);
        const delayHours = parseInt(cpwaSettings.popupDelayHours || 0, 10);
        const delayMinutes = parseInt(cpwaSettings.popupDelayMinutes || 0, 10);
        
        let totalMs = 0;
        
        if (!isNaN(delayDays) && delayDays > 0) {
            totalMs += delayDays * MILLISECONDS_PER_DAY;
        }
        
        if (!isNaN(delayHours) && delayHours > 0) {
            totalMs += delayHours * MILLISECONDS_PER_HOUR;
        }
        
        if (!isNaN(delayMinutes) && delayMinutes > 0) {
            totalMs += delayMinutes * MILLISECONDS_PER_MINUTE;
        }
        
        if (totalMs === 0) {
            totalMs = MILLISECONDS_PER_DAY;
        }
        
        return totalMs;
    }
    
    /**
     * Get detailed dismissal information
     * 
     * @returns {object} Object with dismissal details
     */
    function getPopupDismissalInfo() {
        const timestamp = getPopupDismissalTimestamp();
        const delayMs = getPopupDelayMilliseconds();
        
        if (!timestamp) {
            return {
                dismissed: false,
                timestamp: null,
                delayMs: delayMs,
                nextShowDate: null
            };
        }
        
        const timeSince = Date.now() - timestamp;
        const nextShowDate = timestamp + delayMs;
        
        return {
            dismissed: true,
            timestamp: timestamp,
            timeSince: timeSince,
            delayMs: delayMs,
            nextShowDate: nextShowDate
        };
    }
    
    /**
     * Show loader overlay
     */
    function showLoader() {
        if (loaderOverlay) {
            loaderOverlay.classList.remove('cpwa-hidden');
            loaderOverlay.classList.add('cpwa-visible');
            document.body.style.overflow = 'hidden';
        }
    }
    
    /**
     * Hide loader overlay
     */
    function hideLoader() {
        if (loaderOverlay) {
            setTimeout(() => {
                loaderOverlay.classList.remove('cpwa-visible');
                loaderOverlay.classList.add('cpwa-hidden');
                document.body.style.overflow = '';
                isInstalling = false;
            }, 1000);
        }
    }
    
    /**
     * Show iOS modal
     */
    function showIOSModal() {
        if (iosModal) {
            iosModal.classList.remove('cpwa-hidden');
            iosModal.classList.add('cpwa-visible');
            document.body.style.overflow = 'hidden';
        }
    }
    
    /**
     * Close iOS modal
     */
    function closeIOSModal() {
        if (iosModal) {
            iosModal.classList.remove('cpwa-visible');
            iosModal.classList.add('cpwa-hidden');
            document.body.style.overflow = '';
        }
    }
    
    /**
     * Show success message
     */
    function showSuccessMessage() {
        const successDiv = document.createElement('div');
        successDiv.className = 'cpwa-success-notification';
        successDiv.innerHTML = `
            <svg class="cpwa-success-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
                <polyline points="22 4 12 14.01 9 11.01"></polyline>
            </svg>
            <span>${cpwaSettings.installedText || 'App Installed!'}</span>
        `;
        
        document.body.appendChild(successDiv);
        
        setTimeout(() => {
            successDiv.classList.add('cpwa-visible');
        }, 100);
        
        setTimeout(() => {
            successDiv.classList.remove('cpwa-visible');
            setTimeout(() => {
                if (successDiv.parentNode) {
                    successDiv.parentNode.removeChild(successDiv);
                }
            }, 300);
        }, 3000);
    }
    
    /**
     * Check if app is running in standalone mode (installed)
     * Detects installed state across Android, iOS, and desktop platforms
     */
    function isStandalone() {
        const isStandaloneMedia = window.matchMedia('(display-mode: standalone)').matches;
        const isMinimalUI = window.matchMedia('(display-mode: minimal-ui)').matches;
        const isIOSStandalone = window.navigator.standalone === true;
        const isAndroidApp = document.referrer.includes('android-app://');
        const isTWA = window.navigator.userAgent.includes('TWA') || 
                      window.location.protocol === 'android-app:';
        
        return (
            isStandaloneMedia ||
            isMinimalUI ||
            isIOSStandalone ||
            isAndroidApp ||
            isTWA
        );
    }
    
    /**
     * iOS detection with iPadOS support
     */
    function isIOS() {
        const userAgent = window.navigator.userAgent.toLowerCase();
        const isIPhone = /iphone/.test(userAgent);
        const isIPod = /ipod/.test(userAgent);
        const isIPad = /ipad/.test(userAgent);
        const isIPadOS = /mac/.test(userAgent) && 'ontouchend' in document;
        
        return isIPhone || isIPod || isIPad || isIPadOS;
    }
    
    /**
     * Check if device is Android
     */
    function isAndroid() {
        const userAgent = window.navigator.userAgent.toLowerCase();
        return /android/.test(userAgent);
    }
    
    window.addEventListener('beforeinstallprompt', (e) => {
        if (isStandalone()) {
            return;
        }
        
        e.preventDefault();
        deferredPrompt = e;
        
        if (document.readyState !== 'loading' && installPopup && !isStandalone()) {
            installPopup.classList.remove('cpwa-hidden');
            installPopup.classList.add('cpwa-visible');
            document.body.style.overflow = 'hidden';
        }
    });
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
})();
