<?php
/**
 * CSS Generation Functions
 *
 * @package ChristmasPopUpPro
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Convert hex color to RGB
 */
function cpp_hex_to_rgb( $hex ) {
	$hex = str_replace( '#', '', $hex );
	$r = hexdec( substr( $hex, 0, 2 ) );
	$g = hexdec( substr( $hex, 2, 2 ) );
	$b = hexdec( substr( $hex, 4, 2 ) );
	return [ 'r' => $r, 'g' => $g, 'b' => $b ];
}

/**
 * Generate accent CSS
 */
function cpp_generate_accent_css( $settings ) {
	$accent_enable = isset( $settings['accent_enable'] ) ? absint( $settings['accent_enable'] ) : 1;
	$accent_size = isset( $settings['accent_size'] ) ? absint( $settings['accent_size'] ) : 80;
	$accent_opacity = isset( $settings['accent_opacity'] ) ? floatval( $settings['accent_opacity'] ) : 0.2;
	$accent_blur = isset( $settings['accent_blur'] ) ? absint( $settings['accent_blur'] ) : 10;
	$accent_position = isset( $settings['accent_position'] ) ? esc_attr( $settings['accent_position'] ) : 'top-right';
	
	// Position CSS
	$position_css = '';
	switch ( $accent_position ) {
		case 'top-left':
			$position_css = 'top: 15px; left: 15px; right: auto; bottom: auto;';
			break;
		case 'top-right':
			$position_css = 'top: 15px; right: 15px; left: auto; bottom: auto;';
			break;
		case 'bottom-left':
			$position_css = 'bottom: 15px; left: 15px; right: auto; top: auto;';
			break;
		case 'bottom-right':
			$position_css = 'bottom: 15px; right: 15px; left: auto; top: auto;';
			break;
	}
	
	// Generate glass morphism accent CSS
	$accent_css = '';
	if ( $accent_enable ) {
		$accent_css = "
		.cpp-modal::before {
			content: '';
			position: absolute;
			{$position_css}
			width: {$accent_size}px;
			height: {$accent_size}px;
			background: rgba(255, 255, 255, {$accent_opacity});
			backdrop-filter: blur({$accent_blur}px);
			-webkit-backdrop-filter: blur({$accent_blur}px);
			border-radius: 20px;
			border: 1px solid rgba(255, 255, 255, 0.3);
			box-shadow: 0 8px 32px 0 rgba(0, 0, 0, 0.1);
			pointer-events: none;
			z-index: 1;
		}";
	}
	
	return $accent_css;
}

/**
 * Generate modal background CSS
 */
function cpp_generate_modal_bg_css( $settings ) {
	$bg_start = esc_attr( $settings['bg_grad_start'] );
	$bg_end = esc_attr( $settings['bg_grad_end'] );
	$popup_image = isset( $settings['popup_image'] ) ? esc_url( $settings['popup_image'] ) : '';
	$overlay_color = isset( $settings['overlay_color'] ) ? esc_attr( $settings['overlay_color'] ) : '#f4e2d8';
	$overlay_opacity = isset( $settings['overlay_opacity'] ) ? floatval( $settings['overlay_opacity'] ) : 0.7;
	
	if ( ! empty( $popup_image ) ) {
		$overlay_rgb = cpp_hex_to_rgb( $overlay_color );
		$overlay_rgba = "rgba({$overlay_rgb['r']}, {$overlay_rgb['g']}, {$overlay_rgb['b']}, {$overlay_opacity})";
		return "background: linear-gradient(135deg, {$overlay_rgba} 0%, {$overlay_rgba} 100%), url('{$popup_image}'); background-size: cover; background-position: center; background-repeat: no-repeat;";
	}
	
	return "background: linear-gradient(135deg, {$bg_start} 0%, {$bg_end} 100%);";
}

/**
 * Generate border CSS
 */
function cpp_generate_border_css( $settings ) {
	$border_color = isset( $settings['border_color'] ) ? esc_attr( $settings['border_color'] ) : '';
	$border_width = isset( $settings['border_width'] ) ? absint( $settings['border_width'] ) : 0;
	$border_style = isset( $settings['border_style'] ) ? esc_attr( $settings['border_style'] ) : 'solid';
	$border_radius = isset( $settings['border_radius'] ) ? absint( $settings['border_radius'] ) : 12;
	
	$border_css = '';
	if ( $border_width > 0 && ! empty( $border_color ) ) {
		$border_css = "border: {$border_width}px {$border_style} {$border_color};";
	}
	$border_css .= "border-radius: {$border_radius}px;";
	
	return $border_css;
}

/**
 * Generate shadow CSS
 */
function cpp_generate_shadow_css( $settings ) {
	$shadow_color = isset( $settings['shadow_color'] ) ? esc_attr( $settings['shadow_color'] ) : 'rgba(0, 0, 0, 0.3)';
	$shadow_x = isset( $settings['shadow_x'] ) ? intval( $settings['shadow_x'] ) : 0;
	$shadow_y = isset( $settings['shadow_y'] ) ? intval( $settings['shadow_y'] ) : 10;
	$shadow_blur = isset( $settings['shadow_blur'] ) ? absint( $settings['shadow_blur'] ) : 40;
	$shadow_spread = isset( $settings['shadow_spread'] ) ? intval( $settings['shadow_spread'] ) : 0;
	
	return "box-shadow: {$shadow_x}px {$shadow_y}px {$shadow_blur}px {$shadow_spread}px {$shadow_color};";
}

/**
 * Generate dynamic CSS from settings
 */
function cpp_generate_dynamic_css( $settings ) {
	$text_color = esc_attr( $settings['text_color'] );
	$btn_color = esc_attr( $settings['btn_color'] );
	$btn_hover = esc_attr( $settings['btn_hover'] );
	$btn_hover_text = isset( $settings['btn_hover_text'] ) ? esc_attr( $settings['btn_hover_text'] ) : '#ffffff';
	$btn_secondary_hover_bg = isset( $settings['btn_secondary_hover_bg'] ) ? esc_attr( $settings['btn_secondary_hover_bg'] ) : '';
	
	$bg_css = cpp_generate_modal_bg_css( $settings );
	$border_css = cpp_generate_border_css( $settings );
	$shadow_css = cpp_generate_shadow_css( $settings );
	$accent_css = cpp_generate_accent_css( $settings );
	
	// Secondary button hover background
	$secondary_hover_bg = '';
	if ( ! empty( $btn_secondary_hover_bg ) ) {
		$secondary_hover_bg = "background-color: {$btn_secondary_hover_bg};";
	}
	
	$css = "
		.cpp-overlay {
			backdrop-filter: blur(8px);
			-webkit-backdrop-filter: blur(8px);
		}
		.cpp-modal {
			{$bg_css}
			{$border_css}
			{$shadow_css}
		}
		.cpp-content,
		.cpp-headline,
		.cpp-subheadline,
		.cpp-main-line,
		.cpp-support-line {
			color: {$text_color};
		}
		.cpp-btn-secondary {
			color: {$text_color};
			border-color: {$text_color};
		}
		.cpp-btn-secondary:hover {
			{$secondary_hover_bg}
		}
		.cpp-btn-primary {
			background-color: {$btn_color};
		}
		.cpp-btn-primary:hover {
			background-color: {$btn_hover};
			color: {$btn_hover_text};
		}
		{$accent_css}
	";
	
	return $css;
}

