(function() {
	'use strict';
	
	if (typeof cppSettings === 'undefined') {
		return;
	}
	
	var settings = cppSettings;
	var overlay = null;
	var modal = null;
	var shown = false;
	
	// Storage helpers (localStorage with cookie fallback)
	function setStorage(name, value) {
		try {
			// Ensure value is a string
			var stringValue = String(value);
			localStorage.setItem(name, stringValue);
		} catch (e) {
			// Fallback to cookie if localStorage fails
			setCookie(name, String(value), 365);
		}
	}
	
	function getStorage(name) {
		try {
			var value = localStorage.getItem(name);
			return value;
		} catch (e) {
			// Fallback to cookie if localStorage fails
			return getCookie(name);
		}
	}
	
	function removeStorage(name) {
		try {
			localStorage.removeItem(name);
		} catch (e) {
			deleteCookie(name);
		}
	}
	
	// Cookie helpers (fallback)
	function setCookie(name, value, days) {
		var expires = '';
		if (days) {
			var date = new Date();
			date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
			expires = '; expires=' + date.toUTCString();
		}
		document.cookie = name + '=' + (value || '') + expires + '; path=/; SameSite=Lax';
	}
	
	function getCookie(name) {
		var nameEQ = name + '=';
		var ca = document.cookie.split(';');
		for (var i = 0; i < ca.length; i++) {
			var c = ca[i];
			while (c.charAt(0) === ' ') c = c.substring(1, c.length);
			if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
		}
		return null;
	}
	
	function deleteCookie(name) {
		document.cookie = name + '=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
	}
	
	// Check if should show
	function shouldShow() {
		// Check if popup is scheduled and not expired
		if (settings.disableDate) {
			try {
				var disableTimestamp = new Date(settings.disableDate).getTime();
				var now = new Date().getTime();
				if (!isNaN(disableTimestamp) && now > disableTimestamp) {
					return false; // Popup is expired
				}
			} catch (e) {
				// Invalid date, continue
			}
		}
		
		// Check hide_until (when user clicked "No Thanks")
		// This is the ONLY blocking mechanism - no cookieDays/last_seen blocking
		var hideUntil = getStorage('cpp_hide_until');
		if (hideUntil) {
			var now = new Date().getTime();
			var hideTime = parseInt(hideUntil, 10);
			
			// If parsing failed, try to parse as date string (for old data)
			if (isNaN(hideTime)) {
				var dateParsed = new Date(hideUntil).getTime();
				if (!isNaN(dateParsed)) {
					hideTime = dateParsed;
					// Update storage with correct timestamp format
					setStorage('cpp_hide_until', hideTime.toString());
				} else {
					// Invalid data, clear it
					removeStorage('cpp_hide_until');
					return true; // Allow showing since we cleared invalid data
				}
			}
			
			if (!isNaN(hideTime)) {
				var remainingMs = hideTime - now;
				// Allow 5 second buffer - if delay has essentially expired (within 5 seconds), allow showing
				if (remainingMs > 5000) {
					// Still in hide period - don't show
					return false;
				} else {
					// Expired (or within 5 second buffer) - remove it so popup can show
					removeStorage('cpp_hide_until');
				}
			}
		}
		
		// Mobile check - improved detection
		// Convert to number to handle string values like '0' or '1'
		var mobileDisable = parseInt(settings.mobileDisable, 10) || 0;
		if (mobileDisable === 1) {
			// More reliable mobile detection
			var isMobile = false;
			
			// Check viewport width (use both innerWidth and screen width for reliability)
			var viewportWidth = window.innerWidth || document.documentElement.clientWidth || window.screen.width;
			if (viewportWidth <= 768) {
				isMobile = true;
			}
			
			// Also check for touch capability (mobile devices)
			if (!isMobile && 'ontouchstart' in window || navigator.maxTouchPoints > 0) {
				// Only consider it mobile if viewport is also small
				if (viewportWidth <= 1024) {
					isMobile = true;
				}
			}
			
			if (isMobile) {
				return false;
			}
		}
		
		return true;
	}
	
	// Create pop-up HTML
	function createPopup() {
		overlay = document.createElement('div');
		overlay.className = 'cpp-overlay';
		overlay.id = 'cpp-overlay';
		
		modal = document.createElement('div');
		modal.className = 'cpp-modal';
		
		// Set background image with overlay if available
		if (settings.popupImage && settings.popupImage.trim() !== '') {
			// Convert hex to RGB
			var hex = (settings.overlayColor || '#f4e2d8').replace('#', '');
			var r = parseInt(hex.substr(0, 2), 16);
			var g = parseInt(hex.substr(2, 2), 16);
			var b = parseInt(hex.substr(4, 2), 16);
			var opacity = parseFloat(settings.overlayOpacity || 0.7);
			var overlayRgba = 'rgba(' + r + ', ' + g + ', ' + b + ', ' + opacity + ')';
			
			modal.style.background = 'linear-gradient(135deg, ' + overlayRgba + ' 0%, ' + overlayRgba + ' 100%), url(' + escapeHtml(settings.popupImage) + ')';
			modal.style.backgroundSize = 'cover';
			modal.style.backgroundPosition = 'center';
			modal.style.backgroundRepeat = 'no-repeat';
		}
		
		// Apply text color to all text elements via inline style
		if (settings.textColor) {
			var textColorStyle = 'color: ' + escapeHtml(settings.textColor) + ';';
			modal.style.setProperty('--cpp-text-color', settings.textColor);
		}
		
		// Accent is now handled via CSS, no HTML needed
		modal.innerHTML = 
			'<div class="cpp-content" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' +
				'<h2 class="cpp-headline" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' + escapeHtml(settings.headline) + '</h2>' +
				'<p class="cpp-subheadline" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' + escapeHtml(settings.subheadline) + '</p>' +
				'<p class="cpp-main-line" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' + escapeHtml(settings.mainLine) + '</p>' +
				'<p class="cpp-support-line" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' + escapeHtml(settings.supportLine) + '</p>' +
				'<div class="cpp-buttons">' +
					'<a href="' + escapeHtml(settings.primaryUrl) + '" class="cpp-btn cpp-btn-primary">' + escapeHtml(settings.primaryCta) + '</a>' +
					'<button type="button" class="cpp-btn cpp-btn-secondary cpp-close" style="' + (settings.textColor ? 'color: ' + escapeHtml(settings.textColor) + '; border-color: ' + escapeHtml(settings.textColor) + ';' : '') + '">' + escapeHtml(settings.secondaryCta) + '</button>' +
				'</div>' +
			'</div>';
		
		overlay.appendChild(modal);
		document.body.appendChild(overlay);
		
		// Close handlers
		var closeBtn = modal.querySelector('.cpp-close');
		if (closeBtn) {
			closeBtn.addEventListener('click', closePopup);
		}
		
		// Handle "Buy" button click - set hide_until before navigation
		var buyBtn = modal.querySelector('.cpp-btn-primary');
		if (buyBtn) {
			buyBtn.addEventListener('click', function(e) {
				// Set hide_until same as "No Thanks" - block popup until timer expires
				var hideMins = parseInt(settings.hideMins, 10) || 0;
				var hideHours = parseInt(settings.hideHours, 10) || 0;
				var hideDays = parseInt(settings.hideDays, 10) || 0;
				var totalMs = (hideMins * 60 + hideHours * 60 * 60 + hideDays * 24 * 60 * 60) * 1000;
				
				var now = new Date().getTime();
				var hideUntil = now + totalMs;
				
				// Always set a minimum delay of 1 minute to prevent instant reappearance
				if (totalMs === 0) {
					hideUntil = now + (60 * 1000); // 1 minute minimum
				}
				
				setStorage('cpp_hide_until', hideUntil.toString());
				
				// Allow navigation to proceed
			});
		}
		
		overlay.addEventListener('click', function(e) {
			if (e.target === overlay) {
				closePopup();
			}
		});
	}
	
	function escapeHtml(text) {
		var div = document.createElement('div');
		div.textContent = text;
		return div.innerHTML;
	}
	
	// Show pop-up
	function showPopup() {
		if (shown || !shouldShow()) {
			return;
		}
		
		if (!overlay) {
			createPopup();
		}
		
		document.body.classList.add('cpp-show');
		overlay.classList.add('cpp-show');
		shown = true;
		
		// Don't set last_seen - we want popup to show on every reload until "No Thanks" is clicked
	}
	
	// Close pop-up
	function closePopup() {
		if (!overlay) {
			return;
		}
		
		// Close immediately
		overlay.classList.remove('cpp-show');
		document.body.classList.remove('cpp-show');
		shown = false; // Reset shown flag so it can show again after delay
		
		// Calculate delay in milliseconds
		var hideMins = parseInt(settings.hideMins, 10) || 0;
		var hideHours = parseInt(settings.hideHours, 10) || 0;
		var hideDays = parseInt(settings.hideDays, 10) || 0;
		var totalMs = (hideMins * 60 + hideHours * 60 * 60 + hideDays * 24 * 60 * 60) * 1000;
		
		// Set hide_until timestamp
		var now = new Date().getTime();
		var hideUntil = now + totalMs;
		
		// Always set a minimum delay of 1 minute to prevent instant reappearance
		if (totalMs === 0) {
			hideUntil = now + (60 * 1000); // 1 minute minimum
			totalMs = 60 * 1000;
		}
		
		// Ensure we store as a clean numeric string (timestamp)
		var hideUntilString = Math.floor(hideUntil).toString();
		setStorage('cpp_hide_until', hideUntilString);
		
		// Don't set last_seen - we only use hide_until to block popup
		
		// Schedule a check to show popup again after delay expires
		if (totalMs > 0) {
			setTimeout(function() {
				// Remove hide_until first to allow shouldShow to pass
				removeStorage('cpp_hide_until');
				// Check if delay has expired and show popup
				if (shouldShow() && !shown) {
					// Check if we're still on the same page
					if (document.body && overlay && !overlay.classList.contains('cpp-show')) {
						initExitIntent(); // Re-enable exit intent
						// Try to show after a short delay
						setTimeout(function() {
							if (shouldShow() && !shown) {
								showPopup(); // Show popup after delay expires
							}
						}, 1000);
					}
				}
			}, totalMs + 1000); // Add 1 second buffer
		}
	}
	
	// Exit intent detection
	function initExitIntent() {
		if (!settings.exitIntent || settings.exitIntent === '0') {
			return;
		}
		
		document.addEventListener('mouseleave', function(e) {
			if (!e.toElement && !e.relatedTarget && e.clientY <= 0) {
				if (shouldShow()) {
					showPopup();
				}
			}
		});
	}
	
	// Delay trigger
	function initDelay() {
		var delay = parseInt(settings.delaySeconds, 10) || 5;
		if (delay > 0) {
			setTimeout(function() {
				if (shouldShow() && !shown) {
					showPopup();
				}
			}, delay * 1000);
		}
	}
	
	// Periodic check for re-showing popup after delay expires
	function startPeriodicCheck() {
		setInterval(function() {
			// Only check if popup is not currently shown
			if (!shown) {
				// Check if we have a hide_until that might have expired
				var hideUntil = getStorage('cpp_hide_until');
				if (hideUntil) {
					var now = new Date().getTime();
					var hideTime = parseInt(hideUntil, 10);
					
					// If parsing failed, try to parse as date string (for old data)
					if (isNaN(hideTime)) {
						var dateParsed = new Date(hideUntil).getTime();
						if (!isNaN(dateParsed)) {
							hideTime = dateParsed;
							// Update storage with correct timestamp format
							setStorage('cpp_hide_until', Math.floor(hideTime).toString());
						} else {
							// Invalid data, clear it
							removeStorage('cpp_hide_until');
							hideTime = 0; // Will trigger show
						}
					}
					
					if (!isNaN(hideTime) && now >= hideTime) {
						// Delay expired, remove it first
						removeStorage('cpp_hide_until');
						// Check if we should show now
						if (shouldShow() && !shown) {
							// Re-initialize exit intent
							initExitIntent();
							// Show after a short delay
							setTimeout(function() {
								if (shouldShow() && !shown) {
									var delay = parseInt(settings.delaySeconds, 10) || 5;
									setTimeout(function() {
										if (shouldShow() && !shown) {
											showPopup(); // Show popup after delay expires
										}
									}, delay * 1000);
								}
							}, 500);
						}
					}
				}
			}
		}, 10000); // Check every 10 seconds (more frequent for mobile)
	}
	
	// Initialize
	function init() {
		// Clean up invalid storage data
		var hideUntil = getStorage('cpp_hide_until');
		if (hideUntil) {
			var hideTime = parseInt(hideUntil, 10);
			if (isNaN(hideTime)) {
				// Try parsing as date string (for old data)
				var dateParsed = new Date(hideUntil).getTime();
				if (!isNaN(dateParsed)) {
					// Convert old format to new timestamp format
					setStorage('cpp_hide_until', Math.floor(dateParsed).toString());
				} else {
					// Invalid data, clear it
					removeStorage('cpp_hide_until');
				}
			}
		}
		
		if (shouldShow()) {
			initExitIntent();
			initDelay();
		}
		// Start periodic check for re-showing after delay
		startPeriodicCheck();
	}
	
	// Run when DOM is ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();

