(function($) {
	'use strict';
	
	$(document).ready(function() {
		// Initialize color pickers
		$('.cpp-color-picker').wpColorPicker({
			change: function(event, ui) {
				updatePreview();
			}
		});
		
		// Media Library Image Upload
		var mediaUploader;
		
		// Initialize media uploader when button is clicked
		$(document).on('click', '#cpp-upload-image-btn', function(e) {
			e.preventDefault();
			
			// Check if wp.media is available
			if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
				alert('WordPress Media Library is not loaded. Please refresh the page and try again.');
				return false;
			}
			
			// If media uploader already exists, just open it
			if (mediaUploader) {
				mediaUploader.open();
				return false;
			}
			
			// Create new media uploader
			mediaUploader = wp.media({
				title: 'Choose Popup Image',
				button: {
					text: 'Use this image'
				},
				multiple: false,
				library: {
					type: 'image'
				}
			});
			
			// When an image is selected
			mediaUploader.on('select', function() {
				var attachment = mediaUploader.state().get('selection').first().toJSON();
				if (attachment && attachment.url) {
					$('#popup_image').val(attachment.url);
					$('#cpp-image-preview').html('<img src="' + attachment.url + '" style="max-width: 300px; height: auto; border: 1px solid #ddd; padding: 5px; background: #fff;" />');
					$('#cpp-remove-image-btn').show();
					updatePreview();
				}
			});
			
			// Open the media uploader
			mediaUploader.open();
			
			return false;
		});
		
		// Remove Image
		$('#cpp-remove-image-btn').on('click', function(e) {
			e.preventDefault();
			$('#popup_image').val('');
			$('#cpp-image-preview').html('');
			$(this).hide();
			updatePreview();
		});
		
		// Update preview on input change
		$('#cpp-settings-form input, #cpp-settings-form textarea, #cpp-settings-form select').on('input change', function() {
			updatePreview();
		});
		
		// Save settings
		$('#cpp-save-btn').on('click', function(e) {
			e.preventDefault();
			saveSettings();
		});
		
		// Reset to defaults
		$('#cpp-reset-btn').on('click', function(e) {
			e.preventDefault();
			if (confirm('Are you sure you want to reset all settings to defaults?')) {
				resetDefaults();
			}
		});
		
		// Initial preview
		updatePreview();
	});
	
	function updatePreview() {
		var formData = $('#cpp-settings-form').serializeArray();
		var settings = {};
		
		$.each(formData, function(i, field) {
			if (field.name === 'cpp_nonce') {
				return;
			}
			
			// Handle checkboxes
			if (field.name === 'exit_intent' || field.name === 'mobile_disable') {
				settings[field.name] = field.value === '1' ? 1 : 0;
				return;
			}
			
			settings[field.name] = field.value;
		});
		
		// Explicitly handle checkboxes that might not be in formData if unchecked
		if (!$('#exit_intent').is(':checked')) {
			settings.exit_intent = 0;
		}
		if (!$('#mobile_disable').is(':checked')) {
			settings.mobile_disable = 0;
		}
		
		// Set accent_enable to 0 since we removed the UI
		settings.accent_enable = 0;
		
		$.ajax({
			url: cppAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'cpp_get_preview',
				nonce: cppAdmin.nonce,
				settings: JSON.stringify(settings)
			},
			success: function(response) {
				if (response.success) {
					var iframe = document.getElementById('cpp-preview-iframe');
					if (iframe && iframe.contentDocument) {
						iframe.contentDocument.open();
						iframe.contentDocument.write(response.data.html);
						iframe.contentDocument.close();
					} else if (iframe && iframe.contentWindow) {
						iframe.contentWindow.document.open();
						iframe.contentWindow.document.write(response.data.html);
						iframe.contentWindow.document.close();
					}
				}
			}
		});
	}
	
	function saveSettings() {
		var formData = $('#cpp-settings-form').serializeArray();
		var data = {
			action: 'cpp_save_settings',
			nonce: $('#cpp_nonce').val()
		};
		
		$.each(formData, function(i, field) {
			if (field.name !== 'cpp_nonce') {
				data[field.name] = field.value;
			}
		});
		
		// Explicitly get textarea values
		data.main_line = $('#main_line').val();
		data.support_line = $('#support_line').val();
		
		// Handle checkboxes - ensure they're set to 0 if not checked
		if (!$('#exit_intent').is(':checked')) {
			data.exit_intent = 0;
		} else {
			data.exit_intent = 1;
		}
		if (!$('#mobile_disable').is(':checked')) {
			data.mobile_disable = 0;
		} else {
			data.mobile_disable = 1;
		}
		
		// Ensure accent settings are set to defaults if not present (since we removed the UI)
		if (!data.accent_enable) {
			data.accent_enable = 0;
		}
		
		$.ajax({
			url: cppAdmin.ajaxUrl,
			type: 'POST',
			data: data,
			success: function(response) {
				showMessage(response.success ? 'Settings saved successfully!' : 'Error saving settings.', response.success);
				// Update preview after save
				if (response.success) {
					setTimeout(updatePreview, 500);
				}
			},
			error: function() {
				showMessage('Error saving settings.', false);
			}
		});
	}
	
	function resetDefaults() {
		$.ajax({
			url: cppAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'cpp_reset_defaults',
				nonce: cppAdmin.nonce
			},
			success: function(response) {
				if (response.success) {
					showMessage('Settings reset to defaults!', true);
					// Reload page to show defaults
					setTimeout(function() {
						location.reload();
					}, 1000);
				} else {
					showMessage('Error resetting settings.', false);
				}
			},
			error: function() {
				showMessage('Error resetting settings.', false);
			}
		});
	}
	
	function showMessage(message, success) {
		var $message = $('#cpp-message');
		$message.removeClass('cpp-success cpp-error')
			.addClass(success ? 'cpp-success' : 'cpp-error')
			.text(message)
			.fadeIn();
		
		setTimeout(function() {
			$message.fadeOut();
		}, 3000);
	}
})(jQuery);

